#!/usr/bin/env bash
# 通用轻量防护 & 自动调参脚本 v3
# 支持：
#   - 系统原生 LAMP/LNMP (Debian/Ubuntu/CentOS/Rocky/AlmaLinux 等)
#   - 宝塔面板 (BT Panel)
#   - aaPanel
#
# 菜单：
#   1) 全自动优化（安全 + PHP + MySQL）
#   2) 只做安全（日志 + iptables + fail2ban）
#   3) 只调 PHP（可选版本）
#   4) 只调 MySQL
#   9) 回滚（恢复本脚本的改动）
#   0) 退出脚本

set -e

BACKUP_DIR="/root/server_hardening_backups"

#########################
# 基础信息 / 公共函数
#########################

if [ "$EUID" -ne 0 ]; then
  echo "请用 root 运行本脚本"
  exit 1
fi

if [ -f /etc/os-release ]; then
  . /etc/os-release
  OS_ID=$ID
  OS_VER=$VERSION_ID
else
  OS_ID="unknown"
  OS_VER="unknown"
fi

CORES=$(nproc 2>/dev/null || echo 1)
MEM_KB=$(grep -i MemTotal /proc/meminfo | awk '{print $2}')
MEM_MB=$(( MEM_KB / 1024 ))
MEM_GB=$(( (MEM_MB + 1023) / 1024 ))

mkdir -p "$BACKUP_DIR"

backup_file() {
  local src="$1"
  if [ -f "$src" ]; then
    local ts
    ts=$(date +%Y%m%d%H%M%S)
    local safe
    safe=$(echo "$src" | sed 's#/#_#g')
    local dest="${BACKUP_DIR}/${safe}.${ts}.bak"
    cp "$src" "$dest"
    echo "已备份: $src -> $dest"
  fi
}

restore_latest_backup() {
  local src="$1"
  local safe
  safe=$(echo "$src" | sed 's#/#_#g')
  local latest
  latest=$(ls -1t "${BACKUP_DIR}/${safe}."*.bak 2>/dev/null | head -n 1 || true)
  if [ -n "$latest" ] && [ -f "$latest" ]; then
    cp "$latest" "$src"
    echo "已还原: $src <- $latest"
    return 0
  else
    return 1
  fi
}

#########################
# 自动计算调优参数
#########################

PHP_MAX_CHILDREN=8
PHP_IDLE_TIMEOUT="10s"
MYSQL_BUFFER_POOL="128M"
MYSQL_MAX_CONN=60
MYSQL_LOG_BUFFER="16M"
MYSQL_TABLE_OPEN_CACHE=256

calc_tuning_params() {
  if [ "$MEM_MB" -le 1024 ]; then
    PHP_MAX_CHILDREN=4
    MYSQL_BUFFER_POOL="64M"
    MYSQL_MAX_CONN=30
    MYSQL_LOG_BUFFER="8M"
    MYSQL_TABLE_OPEN_CACHE=128
  elif [ "$MEM_MB" -le 2048 ]; then
    PHP_MAX_CHILDREN=8
    MYSQL_BUFFER_POOL="128M"
    MYSQL_MAX_CONN=50
    MYSQL_LOG_BUFFER="16M"
    MYSQL_TABLE_OPEN_CACHE=256
  elif [ "$MEM_MB" -le 4096 ]; then
    PHP_MAX_CHILDREN=12
    MYSQL_BUFFER_POOL="256M"
    MYSQL_MAX_CONN=80
    MYSQL_LOG_BUFFER="32M"
    MYSQL_TABLE_OPEN_CACHE=512
  elif [ "$MEM_MB" -le 8192 ]; then
    PHP_MAX_CHILDREN=20
    MYSQL_BUFFER_POOL="512M"
    MYSQL_MAX_CONN=120
    MYSQL_LOG_BUFFER="64M"
    MYSQL_TABLE_OPEN_CACHE=1024
  else
    PHP_MAX_CHILDREN=30
    MYSQL_BUFFER_POOL="1024M"
    MYSQL_MAX_CONN=200
    MYSQL_LOG_BUFFER="64M"
    MYSQL_TABLE_OPEN_CACHE=2048
  fi

  local max_by_cpu=$(( CORES * 4 ))
  if [ "$PHP_MAX_CHILDREN" -gt "$max_by_cpu" ]; then
    PHP_MAX_CHILDREN="$max_by_cpu"
  fi
  if [ "$PHP_MAX_CHILDREN" -lt 4 ]; then
    PHP_MAX_CHILDREN=4
  fi
}

#########################
# 安全加固相关函数
#########################

hardening_firewall() {
  echo ">>> 调整防火墙日志 (UFW / firewalld)"

  if command -v ufw >/dev/null 2>&1; then
    echo "UFW: logging off"
    ufw logging off || true
  fi

  if command -v firewall-cmd >/dev/null 2>&1; then
    echo "firewalld: log-denied=off"
    firewall-cmd --permanent --set-log-denied=off || true
    firewall-cmd --reload || true
  fi

  if systemctl is-active --quiet systemd-journald 2>/dev/null; then
    echo "重启 systemd-journald..."
    systemctl restart systemd-journald || true
  fi
}

hardening_iptables() {
  echo ">>> 添加基础 iptables 防护 (SYN/UDP flood)"

  iptables -C INPUT -p tcp --syn -m limit --limit 2/s --limit-burst 6 -j ACCEPT 2>/dev/null || \
  iptables -A INPUT -p tcp --syn -m limit --limit 2/s --limit-burst 6 -j ACCEPT

  iptables -C INPUT -p tcp --syn -j DROP 2>/dev/null || \
  iptables -A INPUT -p tcp --syn -j DROP

  iptables -C INPUT -p udp -m limit --limit 10/s --limit-burst 20 -j ACCEPT 2>/dev/null || \
  iptables -A INPUT -p udp -m limit --limit 10/s --limit-burst 20 -j ACCEPT

  iptables -C INPUT -p udp -j DROP 2>/dev/null || \
  iptables -A INPUT -p udp -j DROP

  if command -v netfilter-persistent >/dev/null 2>&1; then
    netfilter-persistent save || true
  elif command -v service >/dev/null 2>&1 && [ -f /etc/redhat-release ]; then
    service iptables save 2>/dev/null || true
  fi
}

install_fail2ban() {
  echo ">>> 安装 & 启用 fail2ban"

  if [[ "$OS_ID" == "ubuntu" || "$OS_ID" == "debian" ]]; then
    apt-get update -y
    apt-get install -y fail2ban
  elif [[ "$OS_ID" == "centos" || "$OS_ID" == "rocky" || "$OS_ID" == "almalinux" || "$OS_ID" == "rhel" ]]; then
    yum install -y epel-release || true
    yum install -y fail2ban
  else
    echo "未知系统 $OS_ID，跳过 fail2ban 安装"
  fi

  if [ -d /etc/fail2ban ]; then
    if [ ! -f /etc/fail2ban/jail.local ]; then
      cat > /etc/fail2ban/jail.local <<EOF
[DEFAULT]
bantime  = 3600
findtime = 600
maxretry = 5

[sshd]
enabled  = true

[nginx-http-auth]
enabled  = true

[apache-auth]
enabled  = true
EOF
      echo "已创建 /etc/fail2ban/jail.local"
    fi
    systemctl enable fail2ban --now || true
  fi
}

security_hardening() {
  echo "=== 只做安全加固：防火墙日志 + iptables + fail2ban ==="
  hardening_firewall
  hardening_iptables
  install_fail2ban
  echo "=== 安全加固完成 ==="
}

#########################
# PHP 调优相关函数
#########################

PHP_CONFS=()

find_php_confs() {
  PHP_CONFS=()

  # ① 系统原生路径（Debian/Ubuntu）
  if [ -d /etc/php ]; then
    while IFS= read -r -d '' file; do
      PHP_CONFS+=("$file")
    done < <(find /etc/php -type f -name "www.conf" -print0 2>/dev/null)
  fi

  # ② 系统原生路径（CentOS 等）
  if [ -d /etc/php-fpm.d ]; then
    while IFS= read -r -d '' file; do
      PHP_CONFS+=("$file")
    done < <(find /etc/php-fpm.d -maxdepth 1 -type f -name "*.conf" -print0 2>/dev/null)
  fi

  # ③ 宝塔 / aaPanel 路径：/www/server/php/74/etc/php-fpm.conf 等
  if [ -d /www/server/php ]; then
    while IFS= read -r -d '' file; do
      PHP_CONFS+=("$file")
    done < <(find /www/server/php -maxdepth 3 -type f \( -name "php-fpm.conf" -o -name "www.conf" \) -print0 2>/dev/null)
  fi
}

restart_php_for_conf() {
  local conf="$1"

  # ① 宝塔 / aaPanel：/www/server/php/82/etc/php-fpm.conf → /etc/init.d/php-fpm-82
  if echo "$conf" | grep -q "/www/server/php/"; then
    local ver
    ver=$(echo "$conf" | sed -n 's#^/www/server/php/\([0-9]\+\)/.*#\1#p')
    if [ -n "$ver" ]; then
      if [ -x "/etc/init.d/php-fpm-${ver}" ]; then
        echo "重启服务: /etc/init.d/php-fpm-${ver} restart"
        /etc/init.d/php-fpm-"$ver" restart || true
        return
      fi
      local svc="php-fpm-${ver}"
      if systemctl list-unit-files | grep -q "^${svc}.service"; then
        echo "重启服务: ${svc}"
        systemctl restart "$svc" || true
        return
      fi
    fi
  fi

  # ② Debian/Ubuntu：/etc/php/8.2/fpm/... → php8.2-fpm
  if echo "$conf" | grep -q "/etc/php/"; then
    local ver
    ver=$(echo "$conf" | sed -n 's#^/etc/php/\([^/]*\)/fpm/.*#\1#p')
    if [ -n "$ver" ]; then
      local svc="php${ver}-fpm"
      if systemctl list-unit-files | grep -q "^${svc}.service"; then
        echo "重启服务: ${svc}"
        systemctl restart "$svc" || true
        return
      fi
    fi
  fi

  # ③ CentOS 等：通用 php-fpm
  if systemctl list-unit-files | grep -q "^php-fpm.service"; then
    echo "重启服务: php-fpm"
    systemctl restart php-fpm || true
  fi
}

tune_single_php_conf() {
  local conf="$1"
  echo "调整 PHP-FPM 配置: $conf"
  backup_file "$conf"

  sed -ri 's/^\s*pm\s*=.*/pm = ondemand/' "$conf" || true
  sed -ri "s/^\s*pm\.max_children\s*=.*/pm.max_children = ${PHP_MAX_CHILDREN}/" "$conf" || true
  sed -ri "s/^\s*pm\.process_idle_timeout\s*=.*/pm.process_idle_timeout = 10s/" "$conf" || true

  grep -q "^pm =" "$conf" || echo "pm = ondemand" >> "$conf"
  grep -q "^pm.max_children" "$conf" || echo "pm.max_children = ${PHP_MAX_CHILDREN}" >> "$conf"
  grep -q "^pm.process_idle_timeout" "$conf" || echo "pm.process_idle_timeout = 10s" >> "$conf"

  restart_php_for_conf "$conf"
}

php_tuning_menu() {
  calc_tuning_params
  echo "当前自动 PHP 调优参数：pm.max_children=${PHP_MAX_CHILDREN}, idle_timeout=10s"

  find_php_confs
  if [ ${#PHP_CONFS[@]} -eq 0 ]; then
    echo "未找到任何 PHP-FPM 配置文件，无法调优 PHP。"
    return
  fi

  echo "检测到以下 PHP-FPM 配置："
  local idx=1
  for conf in "${PHP_CONFS[@]}"; do
    echo "  $idx) $conf"
    idx=$((idx+1))
  done
  echo "  a) 全部配置都调优"
  echo "  0) 取消并返回上一级菜单"
  echo

  read -rp "请选择要调优的编号（例如：1 或 1 3，全部用 a，取消用 0）: " sel

  # 取消并返回主菜单
  if [ "$sel" = "0" ]; then
    echo "已取消 PHP 调优，返回上一级菜单。"
    return
  fi

  if [ "$sel" = "a" ] || [ "$sel" = "A" ]; then
    for conf in "${PHP_CONFS[@]}"; do
      tune_single_php_conf "$conf"
    done
  else
    for num in $sel; do
      if [[ "$num" =~ ^[0-9]+$ ]] && [ "$num" -ge 1 ] && [ "$num" -le ${#PHP_CONFS[@]} ]; then
        tune_single_php_conf "${PHP_CONFS[$((num-1))]}"
      else
        echo "跳过非法编号: $num"
      fi
    done
  fi

  echo "PHP 调优完成。"
}

#########################
# MySQL 调优相关函数
#########################

MYSQL_CONF_DIR=""
MYSQL_TUNE_FILE=""

detect_mysql_conf_dir() {
  MYSQL_CONF_DIR=""
  MYSQL_TUNE_FILE=""

  # ① 优先使用 conf.d 目录（系统原生）
  if [ -d /etc/mysql/conf.d ]; then
    MYSQL_CONF_DIR="/etc/mysql/conf.d"
    MYSQL_TUNE_FILE="${MYSQL_CONF_DIR}/zz-autotune.cnf"
    return
  fi
  if [ -d /etc/my.cnf.d ]; then
    MYSQL_CONF_DIR="/etc/my.cnf.d"
    MYSQL_TUNE_FILE="${MYSQL_CONF_DIR}/zz-autotune.cnf"
    return
  fi

  # ② 没有 conf.d：使用主配置文件（宝塔 / aaPanel / 常见 Linux）
  for f in /etc/my.cnf /etc/mysql/my.cnf /etc/mysql/mariadb.cnf; do
    if [ -f "$f" ]; then
      MYSQL_CONF_DIR="$(dirname "$f")"
      MYSQL_TUNE_FILE="$f"
      return
    fi
  done
}

restart_mysql() {
  systemctl restart mysql 2>/dev/null || true
  systemctl restart mariadb 2>/dev/null || true
  systemctl restart mysqld 2>/dev/null || true
}

tune_mysql_only() {
  calc_tuning_params
  echo "当前自动 MySQL 调优参数："
  echo "  innodb_buffer_pool_size = ${MYSQL_BUFFER_POOL}"
  echo "  max_connections          = ${MYSQL_MAX_CONN}"
  echo "  innodb_log_buffer_size   = ${MYSQL_LOG_BUFFER}"
  echo "  table_open_cache         = ${MYSQL_TABLE_OPEN_CACHE}"

  detect_mysql_conf_dir
  if [ -z "$MYSQL_TUNE_FILE" ]; then
    echo "未找到 MySQL 配置文件或 conf.d 目录，无法调优 MySQL。"
    return
  fi

  backup_file "$MYSQL_TUNE_FILE"

  # conf.d 方式：单独 zz-autotune.cnf
  if [[ "$MYSQL_TUNE_FILE" == */zz-autotune.cnf ]]; then
    cat > "$MYSQL_TUNE_FILE" <<EOF
[mysqld]
# 自动生成的低内存调优参数 (server_hardening.sh)
innodb_buffer_pool_size = ${MYSQL_BUFFER_POOL}
innodb_log_buffer_size   = ${MYSQL_LOG_BUFFER}
max_connections          = ${MYSQL_MAX_CONN}
table_open_cache         = ${MYSQL_TABLE_OPEN_CACHE}
innodb_buffer_pool_instances = 1
query_cache_size         = 0
query_cache_type         = 0
EOF
  else
    # 主 my.cnf / mariadb.cnf：删除旧块再追加新块
    sed -i '/# BEGIN server_hardening_autotune/,/# END server_hardening_autotune/d' "$MYSQL_TUNE_FILE"
    cat >> "$MYSQL_TUNE_FILE" <<EOF

# BEGIN server_hardening_autotune
[mysqld]
# 自动生成的低内存调优参数 (server_hardening.sh)
innodb_buffer_pool_size = ${MYSQL_BUFFER_POOL}
innodb_log_buffer_size   = ${MYSQL_LOG_BUFFER}
max_connections          = ${MYSQL_MAX_CONN}
table_open_cache         = ${MYSQL_TABLE_OPEN_CACHE}
innodb_buffer_pool_instances = 1
query_cache_size         = 0
query_cache_type         = 0
# END server_hardening_autotune
EOF
  fi

  echo "已写入 MySQL 调优配置: $MYSQL_TUNE_FILE"
  restart_mysql
  echo "MySQL 调优完成。"
}

#########################
# 回滚函数
#########################

rollback_all() {
  echo "=== 开始回滚，尽量恢复本脚本做过的改动 ==="

  echo "--- 回滚防火墙日志设置 ---"
  if command -v ufw >/dev/null 2>&1; then
    echo "UFW: logging on"
    ufw logging on || true
  fi
  if command -v firewall-cmd >/dev/null 2>&1; then
    echo "firewalld: log-denied=all"
    firewall-cmd --permanent --set-log-denied=all || true
    firewall-cmd --reload || true
  fi
  if systemctl is-active --quiet systemd-journald 2>/dev/null; then
    systemctl restart systemd-journald || true
  fi

  echo "--- 删除 iptables 防 flood 规则 ---"
  iptables -D INPUT -p tcp --syn -m limit --limit 2/s --limit-burst 6 -j ACCEPT 2>/dev/null || true
  iptables -D INPUT -p tcp --syn -j DROP 2>/dev/null || true
  iptables -D INPUT -p udp -m limit --limit 10/s --limit-burst 20 -j ACCEPT 2>/dev/null || true
  iptables -D INPUT -p udp -j DROP 2>/dev/null || true
  if command -v netfilter-persistent >/dev/null 2>&1; then
    netfilter-persistent save || true
  elif command -v service >/dev/null 2>&1 && [ -f /etc/redhat-release ]; then
    service iptables save 2>/dev/null || true
  fi

  echo "--- 停用 fail2ban ---"
  if systemctl list-unit-files | grep -q "^fail2ban.service"; then
    systemctl disable fail2ban --now || true
  fi

  echo "--- 还原 PHP-FPM 配置 (www.conf / php-fpm.conf) ---"
  find_php_confs
  if [ ${#PHP_CONFS[@]} -gt 0 ]; then
    for conf in "${PHP_CONFS[@]}"; do
      if restore_latest_backup "$conf"; then
        restart_php_for_conf "$conf"
      fi
    done
  fi

  echo "--- 还原 / 清理 MySQL 调优配置 ---"
  detect_mysql_conf_dir
  if [ -n "$MYSQL_TUNE_FILE" ]; then
    if restore_latest_backup "$MYSQL_TUNE_FILE"; then
      echo "已还原 MySQL 配置备份: $MYSQL_TUNE_FILE"
    else
      if [[ "$MYSQL_TUNE_FILE" == */zz-autotune.cnf ]]; then
        # conf.d 独立文件：直接删
        if [ -f "$MYSQL_TUNE_FILE" ]; then
          echo "删除自动生成的 MySQL 调优文件: $MYSQL_TUNE_FILE"
          rm -f "$MYSQL_TUNE_FILE"
        fi
      else
        # 主 my.cnf：只删除我们插入的 block
        if [ -f "$MYSQL_TUNE_FILE" ]; then
          echo "清理 MySQL 主配置中的自动调优块"
          sed -i '/# BEGIN server_hardening_autotune/,/# END server_hardening_autotune/d' "$MYSQL_TUNE_FILE" || true
        fi
      fi
    fi
    restart_mysql
  fi

  echo "=== 回滚完成。如果还有异常，建议重启服务器再观察。==="
}

#########################
# 全自动模式（安全 + PHP + MySQL）
#########################

full_auto() {
  calc_tuning_params
  echo "=== 全自动优化：安全 + PHP + MySQL ==="
  echo "PHP:  pm.max_children=${PHP_MAX_CHILDREN}, idle_timeout=10s"
  echo "MySQL: innodb_buffer_pool_size=${MYSQL_BUFFER_POOL}, max_connections=${MYSQL_MAX_CONN}"
  echo

  security_hardening
  php_tuning_menu
  tune_mysql_only

  echo "=== 全自动优化完成 ==="
}

#########################
# 菜单
#########################

while true; do
  echo "==============================================="
  echo "   Server Hardening & Auto Tuning Script"
  echo "==============================================="
  echo "系统:   $OS_ID $OS_VER"
  echo "CPU核:  $CORES"
  echo "内存:   ${MEM_MB} MB (~${MEM_GB} GB)"
  echo "-----------------------------------------------"
  echo "请选择操作："
  echo "  1) 全自动优化（安全 + PHP + MySQL）"
  echo "  2) 只做安全（日志 + iptables + fail2ban）"
  echo "  3) 只调 PHP（选择版本）"
  echo "  4) 只调 MySQL"
  echo "  9) 回滚（恢复本脚本的改动）"
  echo "  0) 退出脚本"
  echo

  read -rp "请输入选项 [1/2/3/4/9/0]: " choice

  case "$choice" in
    1)
      full_auto
      ;;
    2)
      security_hardening
      ;;
    3)
      php_tuning_menu
      ;;
    4)
      tune_mysql_only
      ;;
    9)
      rollback_all
      ;;
    0)
      echo "退出脚本。"
      break
      ;;
    *)
      echo "无效选项，请重新输入。"
      ;;
  esac

  echo
  read -rp "按回车键返回菜单..." _tmp
done
